package com.example.sefinsa_app.migrations;

import android.content.ContentValues;
import android.content.Context;
import android.database.Cursor;
import android.database.sqlite.SQLiteDatabase;
import android.database.sqlite.SQLiteOpenHelper;
import android.util.Log;

import com.example.sefinsa_app.models.Aval;
import com.example.sefinsa_app.models.Cliente;
import com.example.sefinsa_app.models.Pago;
import com.example.sefinsa_app.models.Prestamo;

import java.util.ArrayList;
import java.util.List;

public class DatabaseHelper extends SQLiteOpenHelper {

    private static final String DATABASE_NAME = "credigrup.db";
    private static final int DATABASE_VERSION = 30;

    // Nombres de las tablas
    public static final String TABLE_CLIENTES = "clientes";
    public static final String TABLE_AVALES = "avales";
    public static final String TABLE_PRESTAMOS = "prestamos";
    public static final String TABLE_PAGOS = "pagos";
    public static final String TABLE_COLOCADORAS = "colocadoras";
    public static final String TABLE_POBLACIONES = "poblaciones";
    public static final String TABLE_RUTAS = "rutas";
    public static final String TABLE_CONFIGURACION = "configuracion_semanas";
    public static final String TABLE_GASTOS = "gastos";
    public static final String TABLE_REPORTE_CREDITOS = "reporte_creditos";
    public static final String TABLE_REPORTE_DIARIOS_RUTAS = "reporte_diario_rutas";
    public static final String TABLE_FOLIOS = "folios";

    private static DatabaseHelper instance;

    // Constructor privado para implementar el patrón Singleton
    public DatabaseHelper(Context context) {
        super(context, DATABASE_NAME, null, DATABASE_VERSION);
    }

    // Método para obtener la instancia única
    public static synchronized DatabaseHelper getInstance(Context context) {
        if (instance == null) {
            instance = new DatabaseHelper(context.getApplicationContext());
        }
        return instance;
    }

    private boolean isColumnExists(SQLiteDatabase db, String tableName, String columnName) {
        Cursor cursor = db.rawQuery(
                "PRAGMA table_info(" + tableName + ")", null);
        boolean exists = false;
        while (cursor.moveToNext()) {
            String name = cursor.getString(cursor.getColumnIndexOrThrow("name"));
            if (name.equals(columnName)) {
                exists = true;
                break;
            }
        }
        cursor.close();
        return exists;
    }

    @Override
    public void onUpgrade(SQLiteDatabase db, int oldVersion, int newVersion) {
        Log.d("DatabaseHelper", "onUpgrade: oldVersion = " + oldVersion + ", newVersion = " + newVersion);

        if (oldVersion < 24) {
            db.execSQL("DELETE FROM pagos");

            // Agregar la columna 'sinc' a la tabla 'pagos' si no existe
            if (!isColumnExists(db, TABLE_PAGOS, "sinc")) {
                Log.d("DatabaseHelper", "Columna 'sinc' no existe, agregando...");
                db.execSQL("ALTER TABLE " + TABLE_PAGOS + " ADD COLUMN sinc TEXT DEFAULT '0'");
            } else {
                Log.d("DatabaseHelper", "Columna 'sinc' ya existe.");
            }
        }

        if (oldVersion < 24) { // Cambié la versión para evitar conflictos
            Log.d("DatabaseHelper", "Actualizando estructura de la tabla 'avales'...");

            db.execSQL("DELETE FROM avales");

            // Agregar la columna 'sinc' a la tabla 'avales' si no existe
            if (!isColumnExists(db, "avales", "sinc")) {
                Log.d("DatabaseHelper", "Columna 'sinc' no existe, agregando...");
                db.execSQL("ALTER TABLE avales ADD COLUMN sinc TEXT DEFAULT '0'");
            }
        }

        if (oldVersion < 24) {
            Log.d("DatabaseHelper", "Actualizando estructura de la tabla 'clientes'...");

            // Vaciar la tabla sin eliminar su estructura
            Log.d("DatabaseHelper", "Vaciando la tabla 'clientes'...");
            db.execSQL("DELETE FROM clientes");

            // Agregar la columna 'sinc' si no existe
            if (!isColumnExists(db, "clientes", "sinc")) {
                Log.d("DatabaseHelper", "Columna 'sinc' no existe, agregando...");
                db.execSQL("ALTER TABLE clientes ADD COLUMN sinc TEXT DEFAULT '0'");
            }
        }

        if (oldVersion < 24) {
            Log.d("DatabaseHelper", "Actualizando estructura de la tabla 'prestamos'...");

            // Vaciar la tabla sin eliminar su estructura
            Log.d("DatabaseHelper", "Vaciando la tabla 'clientes'...");
            db.execSQL("DELETE FROM prestamos");
        }

        if (oldVersion < 24) {
            // Crear la tabla 'gastos' si no existe
            Log.d("DatabaseHelper", "Creando la tabla 'gastos' si no existe...");
            String createGastosTable = "CREATE TABLE IF NOT EXISTS " + TABLE_GASTOS + " (" +
                    "id INTEGER PRIMARY KEY, " +
                    "otros TEXT, " +
                    "gasolina TEXT, " +
                    "hotel TEXT, " +
                    "alimentos TEXT, " +
                    "casetas TEXT, " +
                    "total TEXT, " +
                    "status TEXT DEFAULT '0') ";
            db.execSQL(createGastosTable);
        }
    }

    @Override
    public void onCreate(SQLiteDatabase db) {
        // Crear todas las tablas aquí
        db.execSQL(createClientesTable());
        db.execSQL(createAvalesTable());
        db.execSQL(createPrestamosTable());
        db.execSQL(createPagosTable());
        db.execSQL(createColocadorasTable());
        db.execSQL(createPoblacionesTable());
        db.execSQL(createRutasTable());
        db.execSQL(createConfiguracionSemanasTable());
        db.execSQL(createReporteCreditosTable());
        db.execSQL(createReporteDiarioDeRutasTable());
        db.execSQL(createGastosTable());
        db.execSQL(createFoliosTable());
    }

    private String createFoliosTable() {
        return "CREATE TABLE IF NOT EXISTS " + TABLE_FOLIOS + " (" +
                "id INTEGER PRIMARY KEY, " +
                "fecha DATE, " +
                "hora TIME, " +
                "cliente_id INTEGER, " +
                "ruta_id INTEGER, " +
                "poblacion_id INTEGER, " +
                "colocadora_id INTEGER, " +
                "monto VARCHAR(12), " +
                "monto_letra TEXT, " +
                "concepto TEXT, " +
                "firma_usuario TEXT, " +
                "firma_cliente TEXT" +
                "sync VARCHAR(1)" +
                ");";
    }

    private String createPagosTable() {
        return "CREATE TABLE " + TABLE_PAGOS + "(" +
                "id integer primary key, " +
                "prestamo_id int, " +
                "cantidad_esperada_pago text, " +
                "cantidad_normal_pagada text, " +
                "cantidad_multa text, " +
                "cantidad_pendiente text, " +
                "cantidad_total_pagada text, " +
                "concepto text, " +
                "fecha_pago text, " +
                "fecha_pago_realizada text, " +
                "folio text, " +
                "semana text, " +
                "balance text, " +
                "empleado_id text, " +
                "status text, " +
                "updated_at text, " +
                "tipo_pago text, " +
                "recuperado text, " +
                "sinc text DEFAULT '0'" +
                ") ";
    }

    private String createReporteCreditosTable() {
        return "CREATE TABLE IF NOT EXISTS " + TABLE_REPORTE_CREDITOS + " (" +
                "id INTEGER PRIMARY KEY, " +
                "poblacion_id TEXT, " +
                "observaciones TEXT, " +
                "creditos_otorgados TEXT, " +
                "total_prestamos TEXT, " +
                "prestamos_nuevos TEXT, " +
                "monto_nuevos TEXT, " +
                "prestamos_renovados TEXT, " +
                "monto_renovados TEXT, " +
                "empleado_id TEXT, " +
                "fecha TEXT, " +
                "fecha_registro TEXT, " +
                "status TEXT DEFAULT '0')";
    }

    private String createReporteDiarioDeRutasTable() {
        return "CREATE TABLE IF NOT EXISTS " + TABLE_REPORTE_DIARIOS_RUTAS + " (" +
                "id INTEGER PRIMARY KEY, " +
                "poblacion_id TEXT, " +
                "estimacion TEXT, " +
                "abonos_semana TEXT, " +
                "recuperacion_semana TEXT, " +
                "recuperacion_vencida TEXT, " +
                "anticipos TEXT, " +
                "total_1 TEXT, " +
                "recuperacion_prestamos TEXT, " +
                "colocacion TEXT, " +
                "prestamo_colocadora TEXT, " +
                "comision TEXT, " +
                "bono_colocadora TEXT, " +
                "total_2 TEXT, " +
                "saldo TEXT, " +
                "pendiente_recuperar TEXT, " +
                "abonos_recuperar TEXT, " +
                "comentario TEXT, " +
                "empleado_id TEXT, " +
                "gastos_id TEXT, " +
                "fecha TEXT, " +
                "fecha_registro TEXT, " +
                "status TEXT DEFAULT '0')";
    }

    private String createConfiguracionSemanasTable() {
        return "CREATE TABLE IF NOT EXISTS " + TABLE_CONFIGURACION + " (" +
                "id integer primary key, " +
                "cantidad int, " +
                "interes text, " +
                "tipo_abono text, " +
                "semana_renovacion int, " +
                "status int) ";
    }

    private String createGastosTable() {
        return "CREATE TABLE IF NOT EXISTS " + TABLE_GASTOS + " (" +
                "id INTEGER PRIMARY KEY, " +
                "otros TEXT, " +
                "gasolina TEXT, " +
                "hotel TEXT, " +
                "alimentos TEXT, " +
                "casetas TEXT, " +
                "total TEXT, " +
                "status TEXT DEFAULT '0') ";
    }

    private String createColocadorasTable() {
        return "CREATE TABLE IF NOT EXISTS " + TABLE_COLOCADORAS + " (" +
                "id INTEGER PRIMARY KEY AUTOINCREMENT, " +
                "nombre_completo TEXT, " +
                "direccion TEXT, " +
                "telefono TEXT, " +
                "status TEXT, " +
                "ruta_id TEXT, " +
                "poblacion_id TEXT, " +
                "created_at TEXT, " +
                "updated_at TEXT)";
    }

    private String createPoblacionesTable() {
        return "CREATE TABLE IF NOT EXISTS " + TABLE_POBLACIONES + " (" +
                "id INTEGER PRIMARY KEY AUTOINCREMENT, " +
                "nombre_poblacion TEXT, " +
                "primer_hora_limite TEXT, " +
                "segunda_hora_limite TEXT, " +
                "primer_dia_cobro TEXT, " +
                "monto_multa TEXT, " +
                "grupo TEXT, " +
                "fecha_inicio TEXT, " +
                "ruta_id TEXT, " +
                "created_at TEXT, " +
                "updated_at TEXT)";
    }

    private String createRutasTable() {
        return "CREATE TABLE IF NOT EXISTS " + TABLE_RUTAS + " (" +
                "id INTEGER PRIMARY KEY AUTOINCREMENT, " +
                "nombre_ruta TEXT, " +
                "empleado_id INTEGER, " +
                "status INTEGER, " +
                "created_at TEXT, " +
                "updated_at TEXT)";
    }

    private String createClientesTable() {
        return "CREATE TABLE " + TABLE_CLIENTES + "(" +
                "id INTEGER UNIQUE, " +
                "nombre_completo TEXT, " +
                "direccion TEXT, " +
                "telefono TEXT, " +
                "garantias TEXT, " +
                "carpeta_comprobantes TEXT, " +
                "carpeta_garantias TEXT, " +
                "otras_referencias TEXT, " +
                "ruta_id TEXT, " +
                "poblacion_id TEXT, " +
                "colocadora_id TEXT, " +
                "created_at TEXT, " +
                "updated_at TEXT, " +
                "update_comprobantes TEXT, " +
                "update_garantias TEXT, " +
                "prestanombre TEXT DEFAULT '0', " +
                "aval_id TEXT)";
    }

    private String createAvalesTable() {
        return "CREATE TABLE " + TABLE_AVALES + "(" +
                "id INTEGER UNIQUE, " +
                "nombre_completo TEXT, " +
                "direccion TEXT, " +
                "telefono TEXT, " +
                "garantias TEXT, " +
                "carpeta_comprobantes TEXT, " +
                "carpeta_garantias TEXT, " +
                "otras_referencias TEXT, " +
                "created_at TEXT, " +
                "updated_at TEXT, " +
                "ruta_id TEXT, " +
                "poblacion_id TEXT, " +
                "colocadora_id TEXT, " +
                "update_comprobantes TEXT, " +      // Nueva columna para los paths de las garantías
                "update_garantias TEXT) ";
    }

    private String createPrestamosTable() {
        return "CREATE TABLE " + TABLE_PRESTAMOS + "(" +
                "id integer primary key, " +
                "cliente_id int, " +
                "direccion_cliente text, " +
                "telefono_cliente text, " +
                "ruta_id int, " +
                "poblacion_id int, " +
                "colocadora_id int, " +
                "aval_id int, " +
                "grupo_poblacion int, " +
                "monto_prestado int, " +
                //"monto_prestado_intereses int, " +
                "pago_semanal int, "  +
                "fecha_prestamo text, " +
                "modalidad_semanas int, "  +
                "numero_tarjeton int, "  +
                "status int, "  +
                "created_at text, " +
                "updated_at text) ";
    }

    public Prestamo getPrestamoById(SQLiteDatabase db, String id) {
        String query = "SELECT id, numero_tarjeton, created_at, updated_at, aval_id FROM " + TABLE_PRESTAMOS + " WHERE id = ?";
        Cursor cursor = db.rawQuery(query, new String[]{id});

        Prestamo prestamo = null;
        if (cursor != null && cursor.moveToFirst()) {
            // Obtener los valores por posición en lugar de por nombre de columna
            String id2 = cursor.getString(0); // ID es la primera columna
            String numero_tarjeton = cursor.getString(1); // Decimotercera columna
            String created_at = cursor.getString(2); // Decimotercera columna
            String updated_at = cursor.getString(3); // Decimotercera columna
            String aval_id = cursor.getString(4); // Decimotercera columna
            //Log.e("getPrestamoById", "Aval obtenido de PRESTAMO aval_id: " + aval_id);
            prestamo = new Prestamo(id2, numero_tarjeton, created_at, updated_at, aval_id);

            cursor.close();
        } else {
            Log.e("getPrestamoById", "No se encontró ningún registro con el id: " + id);
        }

        return prestamo;
    }

    public List<Prestamo> getLast40PrestamosFromSQLite() {
        SQLiteDatabase db = this.getReadableDatabase();
        List<Prestamo> prestamos = new ArrayList<>();

        // Consulta SQL para obtener los últimos 40 registros ordenados por 'updated_at'
        String query = "SELECT id, cliente_id, direccion_cliente, telefono_cliente, ruta_id, poblacion_id, colocadora_id, " +
                "aval_id, grupo_poblacion, monto_prestado, pago_semanal, fecha_prestamo, modalidad_semanas, numero_tarjeton, status, " +
                "created_at, updated_at " +
                "FROM prestamos ORDER BY updated_at DESC";

        Cursor cursor = db.rawQuery(query, null);

        if (cursor.moveToFirst()) {
            do {
                // Usar el constructor parametrizado para inicializar el objeto Aval directamente
                Prestamo prestamo = new Prestamo(
                        cursor.getString(cursor.getColumnIndexOrThrow("id")),
                        cursor.getString(cursor.getColumnIndexOrThrow("cliente_id")),
                        cursor.getString(cursor.getColumnIndexOrThrow("direccion_cliente")),
                        cursor.getString(cursor.getColumnIndexOrThrow("telefono_cliente")),
                        cursor.getString(cursor.getColumnIndexOrThrow("ruta_id")),
                        cursor.getString(cursor.getColumnIndexOrThrow("poblacion_id")),
                        cursor.getString(cursor.getColumnIndexOrThrow("colocadora_id")),
                        cursor.getString(cursor.getColumnIndexOrThrow("aval_id")),
                        cursor.getString(cursor.getColumnIndexOrThrow("grupo_poblacion")),
                        cursor.getString(cursor.getColumnIndexOrThrow("monto_prestado")),
                        cursor.getString(cursor.getColumnIndexOrThrow("pago_semanal")),
                        cursor.getString(cursor.getColumnIndexOrThrow("fecha_prestamo")),
                        cursor.getString(cursor.getColumnIndexOrThrow("modalidad_semanas")),
                        cursor.getString(cursor.getColumnIndexOrThrow("numero_tarjeton")),
                        cursor.getString(cursor.getColumnIndexOrThrow("status")),
                        cursor.getString(cursor.getColumnIndexOrThrow("created_at")),
                        cursor.getString(cursor.getColumnIndexOrThrow("updated_at"))
                );

                // Agregar el objeto a la lista
                prestamos.add(prestamo);
            } while (cursor.moveToNext());
        }

        cursor.close(); // Cerrar el cursor
        db.close();     // Cerrar la base de datos

        return prestamos;
    }

    public List<Aval> getLast30AvalesFromSQLite() {
        SQLiteDatabase db = this.getReadableDatabase();
        List<Aval> avales = new ArrayList<>();

        // Consulta SQL para obtener por 'updated_at'
        String query = "SELECT id, nombre_completo, direccion, telefono, garantias, carpeta_comprobantes, carpeta_garantias, " +
                "otras_referencias, created_at, updated_at, ruta_id, poblacion_id, colocadora_id, update_comprobantes, update_garantias " +
                "FROM avales ORDER BY id DESC";

        Cursor cursor = db.rawQuery(query, null);

        if (cursor.moveToFirst()) {
            do {
                // Usar el constructor parametrizado para inicializar el objeto Aval directamente
                Aval aval = new Aval(
                        cursor.getString(cursor.getColumnIndexOrThrow("id")),
                        cursor.getString(cursor.getColumnIndexOrThrow("nombre_completo")),
                        cursor.getString(cursor.getColumnIndexOrThrow("direccion")),
                        cursor.getString(cursor.getColumnIndexOrThrow("telefono")),
                        cursor.getString(cursor.getColumnIndexOrThrow("garantias")),
                        cursor.getString(cursor.getColumnIndexOrThrow("carpeta_comprobantes")),
                        cursor.getString(cursor.getColumnIndexOrThrow("carpeta_garantias")),
                        cursor.getString(cursor.getColumnIndexOrThrow("otras_referencias")),
                        cursor.getString(cursor.getColumnIndexOrThrow("created_at")),
                        cursor.getString(cursor.getColumnIndexOrThrow("updated_at")),
                        cursor.getString(cursor.getColumnIndexOrThrow("ruta_id")),
                        cursor.getString(cursor.getColumnIndexOrThrow("poblacion_id")),
                        cursor.getString(cursor.getColumnIndexOrThrow("colocadora_id")),
                        cursor.getString(cursor.getColumnIndexOrThrow("update_comprobantes")),
                        cursor.getString(cursor.getColumnIndexOrThrow("update_garantias"))
                );

                // Agregar el objeto a la lista
                avales.add(aval);
            } while (cursor.moveToNext());
        }

        cursor.close(); // Cerrar el cursor
        db.close();     // Cerrar la base de datos

        return avales;
    }

    public void deleteAllReporteDiarioDeRutas() {
        SQLiteDatabase db = this.getWritableDatabase();
        db.execSQL("DELETE FROM reporte_diario_rutas");
        db.close();
    }

    public void deleteAllReporteCreditos() {
        SQLiteDatabase db = this.getWritableDatabase();
        db.execSQL("DELETE FROM reporte_creditos");
        db.close();
    }

    public void deleteAllPrestamos() {
        SQLiteDatabase db = this.getWritableDatabase();
        db.execSQL("DELETE FROM prestamos");
        db.close();
    }

    public void deleteAllClientes() {
        SQLiteDatabase db = this.getWritableDatabase();
        db.execSQL("DELETE FROM clientes");
        db.close();
    }

    public void deleteAllAvales() {
        SQLiteDatabase db = this.getWritableDatabase();
        db.execSQL("DELETE FROM avales");
        db.close();
    }

    public void deleteAllPagos() {
        SQLiteDatabase db = this.getWritableDatabase();
        db.execSQL("DELETE FROM pagos");
        db.close();
    }

    public boolean doesClienteExist(SQLiteDatabase db, String id) {
        String query = "SELECT COUNT(*) FROM " + TABLE_CLIENTES + " WHERE id = ?";
        Cursor cursor = db.rawQuery(query, new String[]{id});
        boolean exists = false;

        if (cursor != null) {
            cursor.moveToFirst();
            exists = cursor.getInt(0) > 0;
            cursor.close();
        }
        return exists;
    }

    public Cliente getClienteById(SQLiteDatabase db, String id) {
        String query = "SELECT id, nombre_completo, direccion, telefono, garantias, carpeta_comprobantes, carpeta_garantias, otras_referencias, ruta_id, poblacion_id, colocadora_id, created_at, updated_at, prestanombre, update_comprobantes, update_garantias, aval_id  FROM " + TABLE_CLIENTES + " WHERE id = ?";
        //Log.d("getClienteById", "Ejecutando consulta: " + query + " con id: " + id);

        Cursor cursor = db.rawQuery(query, new String[]{id});
        Cliente cliente = null;

        if (cursor != null && cursor.moveToFirst()) {
            //Log.d("getClienteById", "Cursor movido a la primera posición. Leyendo datos...");

            // Obtener y loggear cada valor por posición
            String id2 = cursor.getString(0);
            String nombreCompleto = cursor.getString(1);
            String direccion = cursor.getString(2);
            String telefono = cursor.getString(3);
            String garantias = cursor.getString(4);
            String carpetaComprobantes = cursor.getString(5);
            String carpetaGarantias = cursor.getString(6);
            String otrasReferencias = cursor.getString(7);
            String rutaId = cursor.getString(8);
            String poblacionId = cursor.getString(9);
            String colocadoraId = cursor.getString(10);
            String createdAt = cursor.getString(11);
            String updatedAt = cursor.getString(12);
            String prestanombre = cursor.getString(13);
            String update_comprobantes = cursor.getString(14);
            String update_garantias = cursor.getString(15);
            String aval_id = cursor.getString(16);
            Log.e("getClienteById", "Se encontró registro con el aval_id: " + aval_id);
            // Construir el objeto Cliente
            cliente = new Cliente(id2, nombreCompleto, direccion, telefono, garantias,
                    carpetaComprobantes, carpetaGarantias, otrasReferencias,
                    rutaId, poblacionId, colocadoraId, createdAt, updatedAt, prestanombre,update_comprobantes,update_garantias, aval_id);

            cursor.close();
        } else {
            Log.e("getClienteById", "No se encontró ningún registro con el id: " + id);
        }

        //Log.d("getClienteById", "Cliente obtenido: " + (cliente != null ? cliente.toString() : "null"));
        return cliente;
    }

    public List<Cliente> getLast30ClientesFromSQLite() {
        SQLiteDatabase db = this.getReadableDatabase();

        List<Cliente> clientes = new ArrayList<>();
        Cursor cursor = db.rawQuery("SELECT * FROM clientes ORDER BY updated_at DESC LIMIT 30", null);

        if (cursor.moveToFirst()) {
            do {
                Cliente cliente = new Cliente();
                // Mapear columnas del cursor al objeto Pago
                cliente.setId(cursor.getString(cursor.getColumnIndexOrThrow("id"))); // Usa el tipo adecuado (int o String)
                cliente.setNombre_completo(cursor.getString(cursor.getColumnIndexOrThrow("nombre_completo")));
                cliente.setDireccion(cursor.getString(cursor.getColumnIndexOrThrow("direccion")));
                cliente.setTelefono(cursor.getString(cursor.getColumnIndexOrThrow("telefono")));
                cliente.setGarantias(cursor.getString(cursor.getColumnIndexOrThrow("garantias")));
                cliente.setCarpeta_comprobantes(cursor.getString(cursor.getColumnIndexOrThrow("carpeta_comprobantes")));
                cliente.setCarpeta_garantias(cursor.getString(cursor.getColumnIndexOrThrow("carpeta_garantias")));
                cliente.setOtras_referencias(cursor.getString(cursor.getColumnIndexOrThrow("otras_referencias")));
                cliente.setCreated_at(cursor.getString(cursor.getColumnIndexOrThrow("created_at")));
                cliente.setUpdated_at(cursor.getString(cursor.getColumnIndexOrThrow("updated_at")));
                cliente.setRuta_id(cursor.getString(cursor.getColumnIndexOrThrow("ruta_id")));
                cliente.setPoblacion_id(cursor.getString(cursor.getColumnIndexOrThrow("poblacion_id")));
                cliente.setColocadora_id(cursor.getString(cursor.getColumnIndexOrThrow("colocadora_id")));
                cliente.setUpdate_comprobantes_cliente(cursor.getString(cursor.getColumnIndexOrThrow("update_comprobantes")));
                cliente.setUpdate_garantias_cliente(cursor.getString(cursor.getColumnIndexOrThrow("update_garantias")));
                cliente.setAval_id(cursor.getString(cursor.getColumnIndexOrThrow("aval_id")));
                // Completa los demás campos
                clientes.add(cliente);
            } while (cursor.moveToNext());
        }
        cursor.close();
        db.close();

        return clientes;
    }

    public Pago getPagoById(SQLiteDatabase db, String id) {
        String query = "SELECT id, prestamo_id, cantidad_esperada_pago, cantidad_normal_pagada, cantidad_multa, cantidad_pendiente, cantidad_total_pagada, concepto, fecha_pago, fecha_pago_realizada, folio, semana, balance, empleado_id, status, updated_at, tipo_pago, recuperado, sinc  FROM " + TABLE_PAGOS + " WHERE id = ?";
        //Log.d("getPagoById", "Ejecutando consulta: " + query + " con id: " + id);

        Cursor cursor = db.rawQuery(query, new String[]{id});
        Pago pago = null;

        if (cursor != null && cursor.moveToFirst()) {
            //Log.d("getPagoById", "Cursor movido a la primera posición. Leyendo datos...");

            // Obtener y loggear cada valor por posición
            String id2 = cursor.getString(0);
            String prestamo_id = cursor.getString(1);
            String cantidad_esperada_pago = cursor.getString(2);
            String cantidad_normal_pagada = cursor.getString(3);
            String cantidad_multa = cursor.getString(4);
            String cantidad_pendiente = cursor.getString(5);
            String cantidad_total_pagada = cursor.getString(6);
            String concepto = cursor.getString(7);
            String fecha_pago = cursor.getString(8);
            String fecha_pago_realizada = cursor.getString(9);
            String folio = cursor.getString(10);
            String semana = cursor.getString(11);
            String balance = cursor.getString(12);
            String empleado_id = cursor.getString(13);
            String status = cursor.getString(14);
            String updated_at = cursor.getString(15);
            String tipo_pago = cursor.getString(16);
            String recuperado = cursor.getString(17);
            String sinc = cursor.getString(18);

            // Construir el objeto Pago
            pago = new Pago(id2, prestamo_id, cantidad_esperada_pago, cantidad_normal_pagada, cantidad_multa,
                    cantidad_pendiente, cantidad_total_pagada, concepto,
                    fecha_pago, fecha_pago_realizada, folio, semana, balance, empleado_id, status, updated_at, tipo_pago, recuperado, sinc);

            cursor.close();
        } else {
            Log.e("getPagoById", "No se encontró ningún registro con el id: " + id);
        }

        //Log.d("getPagoById", "Pago obtenido: " + (pago != null ? pago.toString() : "null"));
        return pago;
    }

    public List<Pago> getPagosFromDatabase(String prestamoId) {
        List<Pago> pagos = new ArrayList<>();
        SQLiteDatabase db = this.getReadableDatabase();

        // Consulta SQL para seleccionar todos los pagos relacionados con el `prestamo_id`
        String query = "SELECT * FROM " + TABLE_PAGOS + " WHERE prestamo_id = ?";
        Cursor cursor = db.rawQuery(query, new String[]{prestamoId});

        // Iterar sobre los resultados y crear objetos Pago
        if (cursor.moveToFirst()) {
            do {
                // Obtener cada columna directamente por su índice
                String id = cursor.getString(0); // ID es la primera columna
                String prestamoIdColumn = cursor.getString(1); // prestamo_id es la segunda columna
                String cantidadEsperadaPago = cursor.getString(2); // tercera columna
                String cantidadNormalPagada = cursor.getString(3); // cuarta columna
                String cantidadMulta = cursor.getString(4); // quinta columna
                String cantidadPendiente = cursor.getString(5); // sexta columna
                String cantidadTotalPagada = cursor.getString(6); // séptima columna
                String concepto = cursor.getString(7); // octava columna
                String fechaPago = cursor.getString(8); // novena columna
                String fechaPagoRealizada = cursor.getString(9); // décima columna
                String folio = cursor.getString(10); // undécima columna
                String semana = cursor.getString(11); // duodécima columna
                String balance = cursor.getString(12); // decimotercera columna
                String empleadoId = cursor.getString(13); // decimocuarta columna
                String status = cursor.getString(14); // decimoquinta columna
                String updatedAt = cursor.getString(15); // decimosexta columna
                String tipoPago = cursor.getString(16); // decimoséptima columna
                String recuperado = cursor.getString(17); // decimoséptima columna
                String sinc = cursor.getString(18); // decimoséptima columna

                // Crear objeto Pago con los valores obtenidos
                Pago pago = new Pago(id, prestamoIdColumn, cantidadEsperadaPago, cantidadNormalPagada,
                        cantidadMulta, cantidadPendiente, cantidadTotalPagada, concepto,
                        fechaPago, fechaPagoRealizada, folio, semana, balance,
                        empleadoId, status, updatedAt, tipoPago, recuperado, sinc);

                // Agregar a la lista de pagos
                pagos.add(pago);
                Log.d("getPagosFromDatabase", "Pago agregado a la lista: " + pago);
            } while (cursor.moveToNext());
        }
        cursor.close();
        db.close();
        return pagos;
    }

    public List<Pago> getLast100PagosFromSQLite() {
        SQLiteDatabase db = this.getReadableDatabase();

        List<Pago> pagos = new ArrayList<>();
        Cursor cursor = db.rawQuery("SELECT * FROM pagos ORDER BY updated_at DESC limit 8990", null);

        if (cursor.moveToFirst()) {
            do {
                Pago pago = new Pago();
                // Mapear columnas del cursor al objeto Pago
                pago.setId(cursor.getString(cursor.getColumnIndexOrThrow("id"))); // Usa el tipo adecuado (int o String)
                pago.setPrestamo_id(cursor.getString(cursor.getColumnIndexOrThrow("prestamo_id")));
                pago.setCantidad_esperada_pago(cursor.getString(cursor.getColumnIndexOrThrow("cantidad_esperada_pago")));
                pago.setCantidad_normal_pagada(cursor.getString(cursor.getColumnIndexOrThrow("cantidad_normal_pagada")));
                pago.setCantidad_multa(cursor.getString(cursor.getColumnIndexOrThrow("cantidad_multa")));
                pago.setCantidad_pendiente(cursor.getString(cursor.getColumnIndexOrThrow("cantidad_pendiente")));
                pago.setCantidad_total_pagada(cursor.getString(cursor.getColumnIndexOrThrow("cantidad_total_pagada")));
                pago.setConcepto(cursor.getString(cursor.getColumnIndexOrThrow("concepto")));
                pago.setFecha_pago(cursor.getString(cursor.getColumnIndexOrThrow("fecha_pago")));
                pago.setFecha_pago_realizada(cursor.getString(cursor.getColumnIndexOrThrow("fecha_pago_realizada")));
                pago.setFolio(cursor.getString(cursor.getColumnIndexOrThrow("folio")));

                pago.setSemana(cursor.getString(cursor.getColumnIndexOrThrow("semana")));
                pago.setEmpleado_id(cursor.getString(cursor.getColumnIndexOrThrow("empleado_id")));
                pago.setStatus(cursor.getString(cursor.getColumnIndexOrThrow("status")));
                pago.setUpdated_at(cursor.getString(cursor.getColumnIndexOrThrow("updated_at")));
                pago.setTipo_pago(cursor.getString(cursor.getColumnIndexOrThrow("tipo_pago")));
                pago.setRecuperado(cursor.getString(cursor.getColumnIndexOrThrow("recuperado")));
                pago.setSinc(cursor.getString(cursor.getColumnIndexOrThrow("sinc")));
                // Completa los demás campos
                pagos.add(pago);
            } while (cursor.moveToNext());
        }
        cursor.close();
        db.close();

        return pagos;
    }
}
